/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *    CHAIDStarClassifierTree.java
 *    Copyright (C) 2021 ALDAPA Team (http://www.aldapa.eus)
 *    Faculty of Informatics, Donostia, 20018
 *    University of the Basque Country (UPV/EHU), Basque Country
 *
 */

package weka.classifiers.trees.jchaidstar;

import weka.classifiers.trees.jchaid.CHAIDClassifierTree;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instances;
import weka.core.RevisionUtils;
import weka.core.Utils;

/**
 * Class for handling a tree structure for CHAID* algorithm.
 * *************************************************************************************<br/>
 *
 * @author Jes&uacute;s M. P&eacute;rez (txus.perez@ehu.eus)
 * @author Oscar Teixeira (oteixeira001@ikasle.ehu.es)
 * @version $Revision: 1.2 $
 */
public class CHAIDStarClassifierTree extends CHAIDClassifierTree {

  /** for serialization */
  private static final long serialVersionUID = -566548701965318026L;

  /**
   * Constructor for tree structure based on CHAID* algorithm. Stores reference
   * to associated training data at each node.
   *
   * @param toSelectLocModel selection method for local splitting model
   * @param pruneTree true if the tree is to be pruned
   * @param cf the confidence factor for pruning
   * @param raiseTree true if subtree raising is to be performed
   * @param cleanup true to cleanup after build the tree
   * @param collapseTree true if the tree is to be collapsed
   * @throws Exception  if something goes wrong
   */
  public CHAIDStarClassifierTree(CHAIDStarModelSelection toSelectLocModel,
		    boolean pruneTree, float cf,
		    boolean raiseTree,
		    boolean cleanup,
		    boolean collapseTree) throws Exception {
	  super(toSelectLocModel, cleanup);
	  m_pruneTheTree = pruneTree;
	  m_CF = cf;
	  m_subtreeRaising = raiseTree;
	  m_collapseTheTree = collapseTree;
  }

  /**
   * Returns default capabilities of the classifier tree.
   *
   * @return the capabilities of this classifier tree
   */
  @Override
  public Capabilities getCapabilities() {
    Capabilities result = super.getCapabilities();

    result.enable(Capability.NUMERIC_ATTRIBUTES);

    return result;
  }

  /**
   * Method for building a pruneable classifier tree based on CHAID* algorithm.
   *
   * @param data the data for building the tree
   * @throws Exception if something goes wrong
   */
  @Override
  public void buildClassifier(Instances data) throws Exception {

	  // remove instances with missing class
	  data = new Instances(data);
	  data.deleteWithMissingClass();

	  buildTree(data, m_subtreeRaising || !m_cleanup);
	  if (m_collapseTheTree) {
		  collapse();
	  }
	  if (m_pruneTheTree) {
		  prune();
	  }
	  if (m_cleanup) {
		  cleanup(new Instances(data, 0));
	  }
  }

  /**
   * Builds the tree structure.
   *
   * @param data the data for which the tree structure is to be generated.
   * @param keepData is training data to be kept?
   * @throws Exception if something goes wrong
   */
  @Override
  public void buildTree(Instances data, boolean keepData) throws Exception {

    Instances[] localInstances;

    if (keepData) {
      m_train = data;
    }
    m_test = null;
    m_isLeaf = false;
    m_isEmpty = false;
    m_sons = null;
    m_localModel = m_toSelectModel.selectModel(data);
    if (m_localModel.numSubsets() > 1) {
      localInstances = m_localModel.split(data);
      data = null;
      m_sons = new CHAIDStarClassifierTree[m_localModel.numSubsets()];
      for (int i = 0; i < m_sons.length; i++) {
        m_sons[i] = getNewTree(localInstances[i]);
        localInstances[i] = null;
      }
    } else {
      m_isLeaf = true;
      if (Utils.eq(data.sumOfWeights(), 0)) {
        m_isEmpty = true;
      }
      data = null;
    }
  }

  /**
   * Returns a newly created tree.
   *
   * @param data the training data
   * @return the generated tree
   * @throws Exception if something goes wrong
   */
  @Override
  protected CHAIDStarClassifierTree getNewTree(Instances data) throws Exception {

    CHAIDStarClassifierTree newTree =
    		new CHAIDStarClassifierTree((CHAIDStarModelSelection) m_toSelectModel,
    				m_pruneTheTree, m_CF, m_subtreeRaising, m_cleanup, m_collapseTheTree);
    newTree.buildTree(data, m_subtreeRaising || !m_cleanup);

    return newTree;
  }

  /**
   * Returns a newly created tree.
   *
   * @param train the training data
   * @param test the pruning data.
   * @return the generated tree
   * @throws Exception if something goes wrong
   */
  @Override
  protected CHAIDStarClassifierTree getNewTree(Instances train, Instances test)
    throws Exception {

    CHAIDStarClassifierTree newTree =
    		new CHAIDStarClassifierTree((CHAIDStarModelSelection) m_toSelectModel,
    				m_pruneTheTree, m_CF, m_subtreeRaising, m_cleanup, m_collapseTheTree);
    newTree.buildTree(train, test, m_subtreeRaising || !m_cleanup);

    return newTree;
  }

  /**
   * Returns the revision string.
   *
   * @return the revision
   */
  @Override
  public String getRevision() {
    return RevisionUtils.extract("$Revision: 1.1 $");
  }
}
