/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *    CHAIDClassifierTree.java
 *    Copyright (C) 2021 ALDAPA Team (http://www.aldapa.eus)
 *    Faculty of Informatics, Donostia, 20018
 *    University of the Basque Country (UPV/EHU), Basque Country
 *
 */

package weka.classifiers.trees.jchaid;

import weka.classifiers.trees.j48.C45PruneableClassifierTree;
import weka.classifiers.trees.j48.ClassifierTree;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.RevisionUtils;
import weka.core.Utils;

/**
 * Class for handling a tree structure for CHAID algorithm.
 * *************************************************************************************<br/>
 *
 * @author Jes&uacute;s M. P&eacute;rez (txus.perez@ehu.eus)
 * @author Oscar Teixeira (oteixeira001@ikasle.ehu.es)
 * @version $Revision: 1.3 $
 */
public class CHAIDClassifierTree extends C45PruneableClassifierTree {

  /** for serialization */
  private static final long serialVersionUID = 8530147282620306470L;

  /**
   * Constructor for tree structure based on CHAID algorithm. Stores reference
   * to associated training data at each node.
   *
   * @param toSelectLocModel selection method for local splitting model
   * @param cleanup true to cleanup after build the tree
   * @throws Exception  if something goes wrong
   */
  public CHAIDClassifierTree(CHAIDModelSelection toSelectLocModel,
		    boolean cleanup) throws Exception {
    super(toSelectLocModel, false, 0.25f, false, cleanup, false);
  }

  /**
   * Returns default capabilities of the classifier tree.
   *
   * @return the capabilities of this classifier tree
   */
  @Override
  public Capabilities getCapabilities() {
    Capabilities result = super.getCapabilities();

    result.disable(Capability.NUMERIC_ATTRIBUTES);

    return result;
  }

  /**
   * Method for building a classifier tree based on CHAID algorithm.
   *
   * @param data the data for building the tree
   * @throws Exception if something goes wrong
   */
  @Override
  public void buildClassifier(Instances data) throws Exception {

    // remove instances with missing class
    data = new Instances(data);
    data.deleteWithMissingClass();

    buildTree(data, !m_cleanup);
    if (m_cleanup) {
      cleanup(new Instances(data, 0));
    }
  }

  /**
   * Builds the tree structure.
   *
   * @param data the data for which the tree structure is to be generated.
   * @param keepData is training data to be kept?
   * @throws Exception if something goes wrong
   */
  @Override
  public void buildTree(Instances data, boolean keepData) throws Exception {

    Instances[] localInstances;

    if (keepData) {
      m_train = data;
    }
    m_test = null;
    m_isLeaf = false;
    m_isEmpty = false;
    m_sons = null;
    m_localModel = m_toSelectModel.selectModel(data);
    if (m_localModel.numSubsets() > 1) {
      localInstances = m_localModel.split(data);
      data = null;
      m_sons = new CHAIDClassifierTree[m_localModel.numSubsets()];
      for (int i = 0; i < m_sons.length; i++) {
        m_sons[i] = getNewTree(localInstances[i]);
        localInstances[i] = null;
      }
    } else {
      m_isLeaf = true;
      if (Utils.eq(data.sumOfWeights(), 0)) {
        m_isEmpty = true;
      }
      data = null;
    }
  }

  /**
   * Returns a newly created tree.
   *
   * @param data the training data
   * @return the generated tree
   * @throws Exception if something goes wrong
   */
  @Override
  protected CHAIDClassifierTree getNewTree(Instances data) throws Exception {

    CHAIDClassifierTree newTree = new CHAIDClassifierTree((CHAIDModelSelection) m_toSelectModel,
    		m_cleanup);
    newTree.buildTree(data, !m_cleanup);

    return newTree;
  }

  /**
   * Returns a newly created tree.
   *
   * @param train the training data
   * @param test the pruning data.
   * @return the generated tree
   * @throws Exception if something goes wrong
   */
  @Override
  protected CHAIDClassifierTree getNewTree(Instances train, Instances test)
    throws Exception {

    CHAIDClassifierTree newTree = new CHAIDClassifierTree((CHAIDModelSelection) m_toSelectModel,
    		m_cleanup);
    newTree.buildTree(train, test, !m_cleanup);

    return newTree;
  }

  /**
   * Classifies an instance.
   *
   * @param instance the instance to classify
   * @return the classification
   * @throws Exception if something goes wrong
   */
  @Override
  public double classifyInstance(Instance instance) throws Exception {

    double maxProb = -1;
    double currentProb;
    int maxIndex = 0;
    int j;

    for (j = 0; j < instance.numClasses(); j++) {
      currentProb = getProbs(j, instance);
      if (currentProb == -1)
        break;
      if (Utils.gr(currentProb, maxProb)) {
        maxIndex = j;
        maxProb = currentProb;
      }
    }

    if (maxProb == -1) {
      System.out.println("CHAIDClassifierTree.classifyInstance: Unable to classify this instance!");
      return Utils.missingValue();
    }

    return maxIndex;
  }

  /**
   * Returns class probabilities for a weighted instance.
   *
   * @param instance the instance to get the distribution for
   * @param useLaplace whether to use laplace or not
   * @return the distribution
   * @throws Exception if something goes wrong
   */
  @Override
  public final double[] distributionForInstance(Instance instance,
    boolean useLaplace) throws Exception {

    double[] doubles = new double[instance.numClasses()];

    for (int i = 0; i < doubles.length; i++) {
      if (!useLaplace) {
        doubles[i] = getProbs(i, instance);
      } else {
        doubles[i] = getProbsLaplace(i, instance);
      }
    }

    return doubles;
  }

  /**
   * Help method for computing class probabilities of a given instance but
   * without weight since CHAID does not modify the weight of an instance
   * during the classification.
   * Method just done to make program easier to understand.
   *
   * @param classIndex the class index
   * @param instance the instance to compute the probabilities for
   * @return the laplace probs
   * @throws Exception if something goes wrong
   */
  private double getProbsLaplace(int classIndex, Instance instance)
    throws Exception {

    if (m_isLeaf) {
      return getLocalModel().classProbLaplace(classIndex, instance, -1);
    } else {
      int treeIndex = getLocalModel().whichSubset(instance);
      if (treeIndex == -1) {
        return -1;
      } else {
        if (son(treeIndex).m_isEmpty) {
          return getLocalModel().classProbLaplace(classIndex, instance, treeIndex);
        } else {
          return son(treeIndex).getProbsLaplace(classIndex, instance);
        }
      }
    }
  }

  /**
   * Help method for computing class probabilities of a given instance but
   * without weight since CHAID does not modify the weight of an instance
   * during the classification.
   * Method just done to make program easier to understand.
   *
   * @param classIndex the class index
   * @param instance the instance to compute the probabilities for
   * @return the probs
   * @throws Exception if something goes wrong
   */
  private double getProbs(int classIndex, Instance instance)
    throws Exception {

    if (m_isLeaf) {
      return getLocalModel().classProb(classIndex, instance, -1);
    } else {
      int treeIndex = getLocalModel().whichSubset(instance);
      if (treeIndex == -1) {
        return -1;
      } else {
        if (son(treeIndex).m_isEmpty) {
          return getLocalModel().classProb(classIndex, instance, treeIndex);
        } else {
          return son(treeIndex).getProbs(classIndex, instance);
        }
      }
    }
  }

  /**
   * Method just exists to make program easier to read.
   */
  protected CHAIDClassifierTree son(int index) {

    return (CHAIDClassifierTree) m_sons[index];
  }

  /**
   * Return the list of attributes treated as ordinal
   * @return list of ordinal attributes
   */
  public String toStringOrdinalAttributesList() {
	  String st;
	  st = "";
	  for(int i_att=0; i_att < m_train.numAttributes(); i_att++) {
		  if(((CHAIDModelSelection)m_toSelectModel).isOrdinalAtt(i_att)){
			  if(st.isEmpty())
				  st = m_train.attribute(i_att).name();
			  else
				  st += ", " + m_train.attribute(i_att).name();
		  }
	  }
	  return st;
  }
  
  /**
   * Returns source code for the tree as an if-then statement. The class is
   * assigned to variable "p", and assumes the tested instance is named "i". The
   * results are returned as two stringbuffers: a section of code for assignment
   * of the class, and a section of code containing support code (eg: other
   * support methods).
   * 
   * @param className the classname that this static classifier has
   * @return an array containing two stringbuffers, the first string containing
   *         assignment code, and the second containing source for support code.
   * @throws Exception if something goes wrong
   */
  public StringBuffer[] toSource(String className) throws Exception {

    StringBuffer[] result = new StringBuffer[2];
    if (m_isLeaf) {
      result[0] = new StringBuffer("    p = "
          + m_localModel.distribution().maxClass(0) + ";\n");
      result[1] = new StringBuffer("");
    } else {
      StringBuffer text = new StringBuffer();
      StringBuffer atEnd = new StringBuffer();

      long printID = ClassifierTree.nextID();

      text.append("  static double N")
      .append(Integer.toHexString(m_localModel.hashCode()) + printID)
      .append("(Object []i) {\n").append("    double p = Double.NaN;\n");

      int idx;
      int missingCurrentIndex = 0;
      // if the split doesn't take into account missing values,
      // then first ask if the value is missing (because the object is null)
      if (!((CHAIDSplit)m_localModel).hasMissingValues()) {
        text.append("    if (" + m_localModel.sourceExpression(-1, m_train)
            + ") {\n");
        text.append("      return p;\n    }\n");
      } else
        missingCurrentIndex = ((CHAIDSplit)m_localModel).getMissingCurrentIndex(); 
      
      for (int i = 0; i < m_sons.length; i++) {
        // if the split takes into account missing values,
        // then the son which contains this value has to be the first 
        if ( i == 0 ) {
          text.append("    ");
          idx = missingCurrentIndex;
        } else {
          text.append("else ");
          if ( i <= missingCurrentIndex )
            idx = i - 1;
          else
            idx = i;
        }
        text.append("if (" + m_localModel.sourceExpression(idx, m_train)
            + ") {\n");
        if (m_sons[idx].isLeaf()) {
          text.append("        p = " + m_localModel.distribution().maxClass(idx)
              + ";\n");
        } else {
          StringBuffer[] sub = m_sons[idx].toSource(className);
          text.append(sub[0]);
          atEnd.append(sub[1]);
        }
        text.append("    } ");
        if (i == m_sons.length - 1) {
          text.append('\n');
        }
      }

      text.append("    return p;\n  }\n");

      result[0] = new StringBuffer("    p = " + className + ".N");
      result[0].append(Integer.toHexString(m_localModel.hashCode()) + printID)
      .append("(i);\n");
      result[1] = text.append(atEnd);
    }
    return result;
  }

  /**
   * Returns the revision string.
   *
   * @return the revision
   */
  @Override
  public String getRevision() {
    return RevisionUtils.extract("$Revision: 1.1 $");
  }
}
