/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *    JCHAID.java
 *    Copyright (C) 2021 ALDAPA Team (http://www.aldapa.eus)
 *    Faculty of Informatics, Donostia, 20018
 *    University of the Basque Country (UPV/EHU), Basque Country
 *
 */

package weka.classifiers.trees;

import java.util.Collections;
import java.util.Enumeration;
import java.util.Vector;

import weka.classifiers.Sourcable;
import weka.classifiers.trees.jchaid.CHAIDClassifierTree;
import weka.classifiers.trees.jchaid.CHAIDModelSelection;
import weka.core.AdditionalMeasureProducer;
import weka.core.Attribute;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Drawable;
import weka.core.Instances;
import weka.core.Matchable;
import weka.core.Option;
import weka.core.OptionHandler;
import weka.core.PartitionGenerator;
import weka.core.Range;
import weka.core.Summarizable;
import weka.core.TechnicalInformation;
import weka.core.TechnicalInformation.Field;
import weka.core.TechnicalInformation.Type;
import weka.gui.ProgrammaticProperty;
import weka.core.TechnicalInformationHandler;
import weka.core.Utils;

/**
 * <!-- globalinfo-start --> Class for generating a decision tree based on the CHAID algorithm.
 * Unlike C4.5 algorithm (J48 class), CHAID uses chi-squared, &#935;&sup2;, to measure the correlation between
 * the attributes and the class and it only handles discrete variables, although distinguishing
 * between nominal and ordinal behavior.
 * For more information, see<br/>
 * <br/>
 * G. V. Kass. "An Exploratory Technique for Investigating Large Quantities of 
 * Categorical Data". Journal of the Royal Statistical Society - Series C (Applied Statistics) (1980), 29(2), pp 119-127.
 * <a href="http://www.jstor.org/stable/2986296" target="_blank">http://www.jstor.org/stable/2986296</a>
 * <p/>
 * <!-- globalinfo-end -->
 * 
 * <!-- technical-bibtex-start --> BibTeX:
 * 
 * <pre>
 * &#64;article{Kass1980,
 *    author = "G. V. Kass",
 *    title = "An Exploratory Technique for Investigating Large Quantities of Categorical Data",
 *    journal = "Journal of the Royal Statistical Society. Series C (Applied Statistics)",
 *    year = "1980",
 *    volume = "29 (2)",
 *    pages = "119-127",
 *    abstract = "The technique set out in the paper, CHAID, is an offshoot of AID
 *        (Automatic Interaction Detection) designed for a categorized dependent
 *        variable. Some important modifications which are relevant to standard
 *        AID include: built-in significance testing with the consequence of
 *        using the most significant predictor (rather than the most explanatory),
 *        multi-way splits (in contrast to binary) and a new type of predictor
 *        which is especially useful in handling missing information.",
 *    url = "http://www.jstor.org/stable/2986296"
 * }
 * </pre>
 * <p/>
 * <!-- technical-bibtex-end -->
 * 
 * <!-- options-start -->
 * Valid options are: <p/>
 * 
 * J48 options <br/>
 * =========== <br/>
 *
 * <pre>
 * -M &lt;minimum number of instances&gt;
 *  Set minimum number of instances per leaf.
 *  (default 2)
 *  </pre>
 *  
 * <pre>
 * -L
 *  Do not clean up after the tree has been built.
 * </pre>
 * 
 * CHAID options <br/>
 * ============= <br/>
 *
 * <pre> -CH-A &lt;attribute significance level&gt;
 *  Set the significance level for the selection of the attribute to split a node.
 *  (default 0.05)</pre>
 *
 * <pre> -CH-M &lt;merge-split significance level&gt;
 *  Set the significance level for the quest of the best combination of values of attributes.
 *  (default 0.05)</pre>
 *  
 * <pre> -CH-S
 *  Look for the best binary split after merging 3 or more categories
 *  This process could add a considerable latency and that is why it is optional.
 *  (default true)</pre>
 *  
 * <pre> -CH-N &lt;minimum number of instances to split a node&gt;
 *  Set minimum number of instances to consider a node to be split.
 *  (default 3)</pre>
 *  
 * <pre> -CH-O &lt;att1,att2-att4,...&gt;
 *  Specifies list of attribute indexes to set as ordinal. 'First' and 'last' are valid indexes.
 *  Warning: The list of attributes includes the class!
 *  Warning: if XRFF file was used, this option will be ignored!
 *  (default none)</pre>
 * 
 * <!-- options-end -->
 * 
 * @author Jes&uacute;s M. P&eacute;rez (txus.perez@ehu.eus)
 * @author Oscar Teixeira (oteixeira001@ikasle.ehu.es)
 * @version $Revision: 1.2 $
 */
public class JCHAID extends J48 implements OptionHandler, Drawable,
Matchable, Sourcable, Summarizable,
AdditionalMeasureProducer, TechnicalInformationHandler, PartitionGenerator {

	/** for serialization */
	private static final long serialVersionUID = 8132271591300389544L;

	/** Significance level for the selection of the attribute to split a node.
	 *  Based on the the filter MergeNominalValues (package weka.filters.supervised.attribute)).
	 *  However, most implementations of CHAID algorithm distinguish between two significance levels:
	 *  One for the selection of the attributes and another one for the quest of the best combination
	 *  of the categories of an attribute  
	 */
	protected double m_CHsigLevelAtt = 0.05;

	/** Significance level for the quest of the best combination of the categories of an attribute */
	protected double m_CHsigLevelMergeSplit = 0.05;

	/** Indicates if the quest of the best binary split will be done, after merging 3 or more categories
	 *  This process could add a considerable latency and that is why it is optional.
	 */
	protected boolean m_CHsearchBestSplit = true;

	/** Minimum number of instances to split a node */
	protected int m_CHminNumObjSplit = 3;

	/** Stores which attributes are ordinals (monotonic predictors)
	 *  (based on the member m_DiscretizeCols of weka.filters.supervised.attribute.Discretize class) */
	protected Range m_CHordinalAtts = new Range();

	/** Indicates if XRFF format was used */
	protected boolean m_XRFFUsed = false;

	/**
	 * Constructor for JCHAID in order to set the default values
	 *  of the J48 class.
	 */
	public JCHAID(){
		m_unpruned = true;
		m_collapseTree = false;
		m_useMDLcorrection = false;
		m_doNotMakeSplitPointActualValue = true;
	}

	/**
	 * Returns a string describing the classifier
	 * @return a description suitable for
	 * displaying in the explorer/experimenter gui
	 */
	public String globalInfo() {
		return "Class for generating a decision tree based on the CHAID algorithm. Unlike C4.5 "
				+ "algorithm (J48 class), CHAID uses chi-squared, X², to measure the correlation "
				+ "between the attributes and the class and it only handles discrete variables, "
				+ "although distinguishing between nominal and ordinal behaviour. New options are "
				+ "added to the J48 class related to the CHAID algorithm (CH). "
				+ "For more information, see:\n\n"
				+ getTechnicalInformation().toString();
	}

	/**
	 * Returns an instance of a TechnicalInformation object, containing 
	 * detailed information about the technical background of this class,
	 * e.g., paper reference or book this class is based on.
	 * 
	 * @return the technical information about this class
	 */
	public TechnicalInformation getTechnicalInformation() {
		TechnicalInformation 	result;

		result = new TechnicalInformation(Type.ARTICLE);
		result.setValue(Field.AUTHOR, "G. V. Kass");
		result.setValue(Field.YEAR, "1980");
		result.setValue(Field.TITLE, "An Exploratory Technique for Investigating Large Quantities of Categorical Data");
		result.setValue(Field.JOURNAL, "Journal of the Royal Statistical Society. Series C (Applied Statistics)");
		result.setValue(Field.VOLUME, "29");
		result.setValue(Field.NUMBER, "2");
		result.setValue(Field.PAGES, "119-127");
		result.setValue(Field.URL, "http://www.jstor.org/stable/2986296");

		return result;
	}

	/**
	 * Returns default capabilities of the classifier.
	 * 
	 * @return the capabilities of this classifier
	 */
	@Override
	public Capabilities getCapabilities() {
		Capabilities result = super.getCapabilities();

		result.disable(Capability.NUMERIC_ATTRIBUTES);
		result.disable(Capability.DATE_ATTRIBUTES);

		return result;
	}

	/**
	 * Generates the classifier.
	 *
	 * @param instances the data to train the classifier with
	 * @throws Exception if classifier can't be built successfully
	 */
	@Override
	public void buildClassifier(Instances instances) throws Exception {

		getCapabilities().testWithFail(instances);

		double sumOfWeights = instances.sumOfWeights();
		if (Utils.gr(sumOfWeights, 0) && !Utils.eq(sumOfWeights, instances.numInstances())) {
			System.err.println("CHAID cannot use instance weights (sumOfWeights: " + sumOfWeights + ", numInstances: " + instances.numInstances() + ")!");
			throw new Exception("CHAID cannot use instance weights (sumOfWeights: " + sumOfWeights + ", numInstances: " + instances.numInstances() + ")!");
		}

		CHAIDModelSelection modSelection;

		// Prepare the list of ordinal attributes, if exist
		prepareOrdinalAtts(instances);

		modSelection = new CHAIDModelSelection(m_minNumObj, instances,
				m_doNotMakeSplitPointActualValue, 
				m_CHsigLevelAtt, m_CHsigLevelMergeSplit, m_CHsearchBestSplit, m_CHminNumObjSplit, m_CHordinalAtts);

		m_root = new CHAIDClassifierTree(modSelection, !m_noCleanup);

		m_root.buildClassifier(instances);

		modSelection.cleanup();
	}

	/**
	 * Prepares the list of ordinal attributes, extracting metadata from XRFF file, if exists,
	 *  or through the m_CHordinalAtts option.
	 *  If XRFF file is used, this will be priority over m_CHordinalAtts, that is, m_CHordinalAtts will be ignored!
	 *  
	 * @param instances the data to train the classifier with
	 */
	protected void prepareOrdinalAtts(Instances instances) {
		m_CHordinalAtts.setUpper(instances.numAttributes() - 1);
		// Determine list of ordinal attributes (via XRFF file)
		String rangeListXRFF = "";
		for(int i_att=0; i_att < instances.numAttributes(); i_att++) {
			// if properties were supplied for this attribute (via XRFF),
			if(instances.attribute(i_att).getMetadata() != null){
				if(!m_XRFFUsed)
					m_XRFFUsed = true;
				// we get the attribute's ordering set
				if(instances.attribute(i_att).ordering() == Attribute.ORDERING_ORDERED){
					if(rangeListXRFF.isEmpty())
						rangeListXRFF = Integer.toString(i_att + 1);
					else
						rangeListXRFF += "," + Integer.toString(i_att + 1);
				}
			}
		}
		if(m_XRFFUsed){
			System.out.println("XRFF format was used. CHordinalAttributeIndices (CH-O) option will be ignored!");
			setCHordinalAttributeIndices(rangeListXRFF);
			m_CHordinalAtts.setUpper(instances.numAttributes() - 1);
		}
	}

	/**
	 * Returns an enumeration describing the available options.
	 *
	 * Valid options are: <p/>
	 *
	 * J48 options <br/>
	 * =========== <br/>
	 *
	 * <pre>
	 * -M &lt;minimum number of instances&gt;
	 *  Set minimum number of instances per leaf.
	 *  (default 2)
	 *  </pre>
	 *  
	 * <pre>
	 * -L
	 *  Do not clean up after the tree has been built.
	 * </pre>
	 * 
	 * CHAID options <br/>
	 * ============= <br/>
	 *
	 * <pre> -CH-A &lt;attribute significance level&gt;
	 *  Set the significance level for the selection of the attribute to split a node.
	 *  (default 0.05)</pre>
	 *  
	 * <pre> -CH-M &lt;merge-split significance level&gt;
	 *  Set the significance level for the quest of the best combination of values of attributes.
	 *  (default 0.05)</pre>
	 *  
	 * <pre> -CH-S
	 *  Look for the best binary split after merging 3 or more categories
	 *  This process could add a considerable latency and that is why it is optional.
	 *  (default true)</pre>
	 *  
	 * <pre> -CH-N &lt;minimum number of instances to split a node&gt;
	 *  Set minimum number of instances to consider a node to be split.
	 *  (default 3)</pre>
	 *  
	 * <pre> -CH-O &lt;att1,att2-att4,...&gt;
	 *  Specifies list of attribute indexes to set as ordinal. 'First' and 'last' are valid indexes.
	 *  Warning: The list of attributes includes the class!
	 *  Warning: if XRFF file was used, this option will be ignored!
	 *  (default none)</pre>
	 * 
	 * @return an enumeration of all the available options.
	 */
	@Override
	public Enumeration<Option> listOptions() {

		Vector<Option> newVector = new Vector<Option>();

		// CHAID options
		// =============
		newVector.addElement(new Option("\tSet the significance level for the selection of the attribute to split a node.\n"
				+ "\t(default 0.05)", "CH-A", 1, "-CH-A <attribute significance level>"));

		newVector.addElement(new Option("\tSet the significance level for the quest of the best combination of values of attributes.\n"
				+ "\t(default 0.05)", "CH-M", 1, "-CH-M <merge-split significance level>"));

		newVector.addElement(new Option("\tLook for the best binary split after merging 3 or more categories.", "CH-S", 0, "-CH-S"));

		newVector.addElement(new Option("\tSet minimum number of instances to split a node.\n"
				+"\t(default 3)", "CH-N", 1, "-CH-N <minimum number of instances to split a node>"));

		newVector.addElement(new Option(
				"\tSpecifies list of attribute indexes to set as ordinal. 'First'"
						+ " and 'last' are valid indexes.\n"
						+ "Warning: The list of attributes includes the class!\n"
						+ "\t(default none)", "CH-O", 1, "-CH-O <att1,att2-att4,...>"));

		// J48 options
		// ===========
		newVector.addAll(Collections.list(super.listOptions()));

		return newVector.elements();
	}

	/**
	 * Parses a given list of options.
	 *
	 * <!-- options-start -->
	 * Valid options are: <p/>
	 * 
	 * J48 options <br/>
	 * =========== <br/>
	 *
	 * <pre>
	 * -M &lt;minimum number of instances&gt;
	 *  Set minimum number of instances per leaf.
	 *  (default 2)
	 *  </pre>
	 *  
	 * <pre>
	 * -L
	 *  Do not clean up after the tree has been built.
	 * </pre>
	 *  
	 * CHAID options <br/>
	 * ============= <br/>
	 * 
	 * <pre> -CH-A &lt;attribute significance level&gt;
	 *  Set the significance level for the selection of the attribute to split a node.
	 *  (default 0.05)</pre>
	 *
	 * <pre> -CH-M &lt;merge-split significance level&gt;
	 *  Set the significance level for the quest of the best combination of values of attributes.
	 *  (default 0.05)</pre>
	 *  
	 * <pre> -CH-S
	 *  Look for the best binary split after merging 3 or more categories
	 *  This process could add a considerable latency and that is why it is optional.
	 *  (default true)</pre>
	 *  
	 * <pre> -CH-N &lt;minimum number of instances to split a node&gt;
	 *  Set minimum number of instances to consider a node to be split.
	 *  (default 3)</pre>
	 *  
	 * <pre> -CH-O &lt;att1,att2-att4,...&gt;
	 *  Specifies list of attribute indexes to set as ordinal. 'First' and 'last' are valid indexes.
	 *  Warning: The list of attributes includes the class!
	 *  Warning: if XRFF file was used, this option will be ignored!
	 *  (default none)</pre>
	 * 
	 * <!-- options-end -->
	 * 
	 * @param options the list of options as an array of strings
	 * @throws Exception if an option is not supported
	 */
	@Override
	public void setOptions(String[] options) throws Exception {

		// CHAID options
		// =============
		String CHsigLevelString = Utils.getOption("CH-A", options);
		if (CHsigLevelString.length() != 0) {
			m_CHsigLevelAtt = (new Double(CHsigLevelString)).doubleValue();
			if ((m_CHsigLevelAtt <= 0) || (m_CHsigLevelAtt >= 1)) {
				throw new Exception(
						"Significance level for the selection of attributes has to be greater than zero and smaller "
								+ "than one!");
			}
		} else {
			m_CHsigLevelAtt = 0.05;
		}
		String CHsigLevelMergeSplit = Utils.getOption("CH-M", options);
		if (CHsigLevelMergeSplit.length() != 0) {
			m_CHsigLevelAtt = (new Double(CHsigLevelMergeSplit)).doubleValue();
			if ((m_CHsigLevelMergeSplit <= 0) || (m_CHsigLevelMergeSplit >= 1)) {
				throw new Exception(
						"Significance level to look for the best combination of categories has to be greater than zero and smaller "
								+ "than one!");
			}
		} else {
			m_CHsigLevelMergeSplit = 0.05;
		}

		m_CHsearchBestSplit = Utils.getFlag("CH-S", options);

		String minNumSplitString = Utils.getOption("CH-N", options);
		if (minNumSplitString.length() != 0) {
			m_CHminNumObjSplit = Integer.parseInt(minNumSplitString);
		} else {
			m_CHminNumObjSplit = 3;
		}

		String convertList = Utils.getOption("CH-O", options);
		if (convertList.length() != 0) {
			setCHordinalAttributeIndices(convertList);
		} else {
			setCHordinalAttributeIndices(""); // default: none
		}

		// J48 options
		// ===========
		super.setOptions(options);

		Utils.checkForRemainingOptions(options);
	}

	/**
	 * Gets the current settings of the Classifier.
	 * 
	 * @return an array of strings suitable for passing to setOptions
	 */
	@Override
	public String[] getOptions() {

		Vector<String> options = new Vector<String>();

		// CHAID options
		// =============
		options.add("-CH-A");
		options.add("" + m_CHsigLevelAtt);
		options.add("-CH-M");
		options.add("" + m_CHsigLevelMergeSplit);
		if (m_CHsearchBestSplit)
			options.add("-CH-S");
		options.add("-CH-N");
		options.add("" + m_CHminNumObjSplit);
		if (!getCHordinalAttributeIndices().equals("")) {
			options.add("-CH-O");
			options.add(getCHordinalAttributeIndices());
		}

		// J48 options
		// ===========
		Collections.addAll(options, super.getOptions());

		return options.toArray(new String[0]);
	}

	/**
	 * Returns a description of the classifier.
	 * 
	 * @return a description of the classifier
	 */
	@Override
	public String toString() {

		if (m_root == null) {
			return "No classifier built";
		} else {
			return "JCHAID tree\n------------------\n" +
					toStringOrdinalAttributesList() +
					m_root.toString();
		}
	}

	/**
	 * Return the list of attributes treated as ordinal
	 * @return list of ordinal attributes
	 */
	public String toStringOrdinalAttributesList() {
		String st;
		st = "List of attributes treated as ordinal:\n";
		st += ((CHAIDClassifierTree)m_root).toStringOrdinalAttributesList();
		st += "\n";
		// Add a separator
		char[] ch_line = new char[st.length()];
		for (int i = 0; i < ch_line.length; i++)
			ch_line[i] = '-';
		String line = String.valueOf(ch_line);
		line += "\n";
		st += line;
		return st;
	}

	/**
	 * Returns the tip text for this property
	 *
	 * @return tip text for this property suitable for displaying in the
	 *         explorer/experimenter gui
	 */
	public String CHsigLevelAttTipText() {
		return "The significance level for the selection of the attribute to split a node.";
	}

	/**
	 * Gets the value of m_CHsigLevelAtt.
	 *
	 * @return the value
	 */
	public double getCHsigLevelAtt() {
		return m_CHsigLevelAtt;
	}

	/**
	 * Sets the value of m_CHsigLevelAtt.
	 *
	 * @param sigLevel the value to set
	 */
	public void setCHsigLevelAtt(double sigLevel) {
		m_CHsigLevelAtt = sigLevel;
	}

	/**
	 * Returns the tip text for this property
	 *
	 * @return tip text for this property suitable for displaying in the
	 *         explorer/experimenter gui
	 */
	public String CHsigLevelMergeSplitTipText() {
		return "The significance level for the quest of the best combination of the categories of an attribute.";
	}

	/**
	 * Gets the value of m_CHsigLevelMergeSplit.
	 *
	 * @return the value
	 */
	public double getCHsigLevelMergeSplit() {
		return m_CHsigLevelMergeSplit;
	}

	/**
	 * Sets the value of m_CHsigLevelMergeSplit.
	 *
	 * @param sigLevel the value to set
	 */
	public void setCHsigLevelMergeSplit(double sigLevel) {
		m_CHsigLevelMergeSplit = sigLevel;
	}

	/**
	 * Returns the tip text for this property
	 *
	 * @return tip text for this property suitable for displaying in the
	 *         explorer/experimenter gui
	 */
	public String CHsearchBestSplitTipText() {
		return "Indicates if the quest of the best binary split will be done, after merging 3 or more categories.";
	}

	/**
	 * Gets the value of m_CHsigLevelMergeSplit.
	 *
	 * @return the value
	 */
	public boolean getCHsearchBestSplit() {
		return m_CHsearchBestSplit;
	}

	/**
	 * Sets the value of m_CHsearchBestSplit.
	 *
	 * @param searchBestSplit the value to set
	 */
	public void setCHsearchBestSplit(boolean searchBestSplit) {
		m_CHsearchBestSplit = searchBestSplit;
	}

	/**
	 * Returns the tip text for this property
	 *
	 * @return tip text for this property suitable for displaying in the
	 *         explorer/experimenter gui
	 */
	public String CHminNumObjSplitTipText() {
		return "Set the minimum number of instances to split a node.";
	}

	/**
	 * Gets the value of m_CHsigLevelMergeSplit.
	 *
	 * @return the value
	 */
	public int getCHminNumObjSplit() {
		return m_CHminNumObjSplit;
	}

	/**
	 * Sets the value of m_CHminNumObjSplit.
	 *
	 * @param minNumObjSplit the value to set
	 */
	public void setCHminNumObjSplit(int minNumObjSplit) {
		m_CHminNumObjSplit = minNumObjSplit;
	}

	/**
	 * Returns the tip text for this property
	 *
	 * @return tip text for this property suitable for displaying in the
	 *         explorer/experimenter gui
	 */
	public String CHordinalAttributeIndicesTipText() {
		return "Specify range of attributes to set as ordinal."
				+ " This is a comma separated list of attribute indices, with"
				+ " \"first\" and \"last\" valid values. Specify an inclusive"
				+ " range with \"-\". E.g: \"first-3,5,6-10,last\"."
				+ " Warning: The list of attributes includes the class!"
				+ " Warning: if XRFF file was used, this option will be ignored!"
				+ " (default: none)";
	}

	/**
	 * Gets the current range selection
	 *
	 * @return a string containing a comma separated list of ranges
	 */
	public String getCHordinalAttributeIndices() {

		return m_CHordinalAtts.getRanges();
	}

	/**
	 * Sets which attributes are to be considered as ordinal
	 *
	 * @param rangeList a string representing the list of attributes. Since the
	 *          string will typically come from a user, attributes are indexed
	 *          from 1. <br>
	 *          eg: first-3,5,6-last
	 *          Warning: The list of attributes includes the class!
	 *          Warning: if XRFF file was used, this option will be ignored!
	 * @throws IllegalArgumentException if an invalid range list is supplied
	 */
	public void setCHordinalAttributeIndices(String rangeList) {

		m_CHordinalAtts.setRanges(rangeList);
	}

	/**
	 * Set the value of Seed.
	 * 
	 * @param newSeed Value to assign to Seed.
	 */
	@Override
	@ProgrammaticProperty
	public void setSeed(int newSeed) {
		throw new IllegalArgumentException("This J48 option does not make sense for JCHAID!.");
	}

	/**
	 * Set the value of unpruned. Turns reduced-error pruning off if set.
	 * 
	 * @param v Value to assign to unpruned.
	 */
	@Override
	@ProgrammaticProperty
	public void setUnpruned(boolean v) {
		if (!v)
			throw new IllegalArgumentException("This J48 option does not make sense for JCHAID!.");
		super.setUnpruned(v);
	}

	/**
	 * Set the value of collapseTree.
	 * 
	 * @param v Value to assign to collapseTree.
	 */
	@Override
	@ProgrammaticProperty
	public void setCollapseTree(boolean v) {
		throw new IllegalArgumentException("This J48 option does not make sense for JCHAID!.");
	}

	/**
	 * Set the value of CF.
	 * 
	 * @param v Value to assign to CF.
	 */
	@Override
	@ProgrammaticProperty
	public void setConfidenceFactor(float v) {
		if (!Utils.eq(v, 0.25f))
			throw new IllegalArgumentException("This J48 option does not make sense for JCHAID!.");
		super.setConfidenceFactor(v);
	}

	/**
	 * Set the value of subtreeRaising.
	 * 
	 * @param v Value to assign to subtreeRaising.
	 */
	@Override
	@ProgrammaticProperty
	public void setSubtreeRaising(boolean v) {
		throw new IllegalArgumentException("This J48 option does not make sense for JCHAID!.");
	}

	/**
	 * Set the value of reducedErrorPruning. Turns
	 * unpruned trees off if set.
	 * (This method only accepts the default value of J48)
	 *
	 * @param v  Value to assign to reducedErrorPruning.
	 */
	@Override
	@ProgrammaticProperty
	public void setReducedErrorPruning(boolean v){
		if (v)
			throw new IllegalArgumentException("J48 option not implemented for JCHAID");
		super.setReducedErrorPruning(v);
	}

	/**
	 * Set the value of numFolds.
	 * (This method only accepts the default value of J48)
	 *
	 * @param v  Value to assign to numFolds.
	 */
	@Override
	@ProgrammaticProperty
	public void setNumFolds(int v) {
		if (v != 3)
			throw new IllegalArgumentException("J48 option not implemented for JCHAID");
		super.setNumFolds(v);
	}

	/**
	 * Set the value of binarySplits.
	 * (This method only accepts the default value of J48)
	 *
	 * @param v  Value to assign to binarySplits.
	 */
	@Override
	@ProgrammaticProperty
	public void setBinarySplits(boolean v) {
		if (v)
			throw new IllegalArgumentException("J48 option not implemented for JCHAID");
		super.setBinarySplits(v);
	}

	/**
	 * Set the value of useLaplace.
	 * (This method only accepts the default value of J48)
	 * 
	 * @param newuseLaplace Value to assign to useLaplace.
	 */
	@Override
	@ProgrammaticProperty
	public void setUseLaplace(boolean newuseLaplace) {
		if (newuseLaplace)
			throw new IllegalArgumentException("J48 option not allowed for JCHAID. You can use JCHAIDStar!");
		super.setUseLaplace(newuseLaplace);
	}

	/**
	 * Set the value of useMDLcorrection.
	 *
	 * @param newuseMDLcorrection Value to assign to useMDLcorrection.
	 */
	@Override
	@ProgrammaticProperty
	public void setUseMDLcorrection(boolean newuseMDLcorrection) {
		throw new IllegalArgumentException("J48 option not implemented for JCHAID. It does not make sense!.");
	}

	/**
	 * Sets the value of doNotMakeSplitPointActualValue.
	 * 
	 * @param m_doNotMakeSplitPointActualValue the value to set
	 */
	@Override
	@ProgrammaticProperty
	public void setDoNotMakeSplitPointActualValue(
			boolean m_doNotMakeSplitPointActualValue) {
		throw new IllegalArgumentException("J48 option not implemented for JCHAID. It does not make sense!.");
	}

	/**
	 * Main method for testing this class
	 *
	 * @param argv the command line options
	 */
	public static void main(String[] argv) {
		runClassifier(new JCHAID(), argv);
	}
}
