/*
    Copyright (C) 2016 University of the Basque Country, UPV/EHU.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/** \file
 * Library management functions.
 */

/********************************************
 * Includes                                 *
 ********************************************/
#include <string.h>

#include "globals.h"
#include "cuPoisson.h"

/********************************************
 * Exported functions                       *
 ********************************************/
extern "C" {

/** \ingroup public
 * Allocate host memory that is page-locked. This makes copies between host
 * and device memory faster.
 * \param[out] ptr A pointer to a memory pointer.
 * \param[in] size Allocation size in bytes.
 * \return CUP_SUCCESS, CUP_INVALID_ARGUMENT if \p ptr is NULL or CUP_CUDA_ERROR.
 */
cup_error_t cup_malloc( void** ptr, size_t size )
{
	if( ptr == NULL )
		return CUP_INVALID_ARGUMENT;

	CUDA( cudaMallocHost( ptr, size ) );

	return CUP_SUCCESS;
}

/** \ingroup public
 * Free page-locked memory.
 * \param[in] ptr Pointer to memory to free.
 * \return CUP_SUCCESS, CUP_INVALID_ARGUMENT if \p ptr is NULL or CUP_CUDA_ERROR.
 */
cup_error_t cup_free( void* ptr )
{
	if( ptr == NULL )
		return CUP_INVALID_ARGUMENT;

	CUDA( cudaFreeHost( ptr ) );

	return CUP_SUCCESS;
}

/**	\ingroup public
 * Tests the platform to obtain the number and identifiers of the devices
 * valid to run the library.
 * \param[out] valid_devices A pointer that on success will point to an array
 *  of integers identifying the valid devices. If NULL, it will be ignored.
 * \param[out] num_valid The number of valid devices detected. If NULL,
 *  it will be ignored.
 * \return CUP_SUCCESS, CUP_OUT_OF_HOST_MEM.
 */
cup_error_t cup_get_valid_devices( int** valid_devices, int* num_valid )
{
	int  i,
		 total_devices,
		 _num_valid,
	   * _valid_devices;
	struct cudaDeviceProp props;
	cudaError_t st;

	st = cudaGetDeviceCount( &total_devices );
	if( st == cudaErrorNoDevice )
	{
		if( num_valid != NULL )
			*num_valid = 0;
		if( valid_devices != NULL )
			*valid_devices = NULL;
		return CUP_SUCCESS;
	}
	if( st != cudaSuccess )
		return CUP_CUDA_ERROR;

	if( valid_devices != NULL )
		MALLOC( _valid_devices, total_devices, int );

	for( _num_valid = 0, i = 0; i < total_devices; i++ )
	{
		cudaGetDeviceProperties( &props, i );
		if( props.major >= 2 || (props.major == 1 && props.minor >= 3) )
		{
			if( valid_devices != NULL)
				_valid_devices[_num_valid] = i;
			_num_valid++;
		}
	}

	if( num_valid != NULL)
		*num_valid = _num_valid;
	if( valid_devices != NULL )
		*valid_devices = (int *) realloc( _valid_devices,
		                                  _num_valid * sizeof( int ) );

	return CUP_SUCCESS;
}

/** \ingroup utils
 * Get a descriptive string from an error code of the cuPoisson library.
 *  \param[in] error An error code from the cuPoisson library.
 *  \return A pointer to a string describing the error.
 */
const char* cup_error_string( cup_error_t error )
{
	static const char* error_strings[] =
	{
		"Success.",
		"Unable to initialize cuPoisson library.",
		"cuPoisson library was not initialized.",
		"cuPoisson library was initialized.",
		"No GPU devices found.",
		"No valid GPU devices found.",
		"Out of host memory.",
		"Out of device memory.",
		"Invalid argument.",
		"Thread management error.",
		"CUDA error.",
		"MPI error.",
		"Unknown error"
	};

	if( error >= CUP_UNKNOWN || error < CUP_SUCCESS )
		return error_strings[CUP_UNKNOWN];
	else
		return error_strings[error];
}

} // extern "C"

