/*
    Copyright (C) 2016 University of the Basque Country, UPV/EHU.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/********************************************
 * Includes                                 *
 ********************************************/
#include "utils/alg.cu"

#include <stdlib.h>

#include "test.h"
#include "utils/test_alg.h"

/********************************************
 * Private function prototypes              *
 ********************************************/
static void init_vec( double* v, int N );


/********************************************
 * Public functions                         *
 ********************************************/
int test_src_utils_alg( char* function )
{
	cup_error_t status;
	const int N = 100;
	double  v1[N],
	        v2[N],
	        res[N],
	      * dev_v1,
	      * dev_v2,
	      * dev_res;
	const double ALFA = 1.2345;
	const double EPSILON = 1e-10;
	int i;

	init_vec( v1, N );
	init_vec( v2, N );

	CUDA( cudaMalloc( &dev_v1, N * sizeof( double) ) );
	CUDA( cudaMalloc( &dev_v2, N * sizeof( double) ) );
	CUDA( cudaMalloc( &dev_res, N * sizeof( double) ) );

	CUDA( cudaMemcpy( dev_v1, v1, N * sizeof( double ), cudaMemcpyHostToDevice ) );
	CUDA( cudaMemcpy( dev_v2, v2, N * sizeof( double ), cudaMemcpyHostToDevice ) );

	if( !strcmp( function, "daddvv" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		daddvv( dev_v1, dev_v2, dev_res, N, 0 );

		CUDA( cudaMemcpy( res, dev_res, N * sizeof( double ), cudaMemcpyDeviceToHost ) );

		for( i = 0; i < N; i++ )
			TEST( fabs( res[i] - (v1[i] + v2[i]) ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "dsubvv" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		dsubvv( dev_v1, dev_v2, dev_res, N, 0 );

		CUDA( cudaMemcpy( res, dev_res, N * sizeof( double ), cudaMemcpyDeviceToHost ) );

		for( i = 0; i < N; i++ )
			TEST( fabs( res[i] - (v1[i] - v2[i]) ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "dnorm2v" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		dnorm2v( dev_v1, dev_res, N );

		CUDA( cudaMemcpy( res, dev_res, sizeof( double ), cudaMemcpyDeviceToHost ) );

		res[1] = 0;
		for( i = 0; i < N; i++ )
			res[1] += v1[i] * v1[i];
		TEST( fabs( (res[0] - res[1]) / res[0] ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "ddotp" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		ddotp( dev_v1, dev_v2, dev_res, N );

		CUDA( cudaMemcpy( res, dev_res, sizeof( double ), cudaMemcpyDeviceToHost ) );

		res[1] = 0;
		for( i = 0; i < N; i++ )
			res[1] += v1[i] * v2[i];
		TEST( fabs( (res[0] - res[1]) / res[0] ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "dsaddvv" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		dsaddvv( ALFA, dev_v1, dev_v2, dev_res, N, 0 );

		CUDA( cudaMemcpy( res, dev_res, N * sizeof( double ), cudaMemcpyDeviceToHost ) );

		for( i = 0; i < N; i++ )
			TEST( fabs( res[i] - (v1[i] + ALFA * v2[i]) ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "dssubvv" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		dssubvv( ALFA, dev_v1, dev_v2, dev_res, N, 0 );

		CUDA( cudaMemcpy( res, dev_res, N * sizeof( double ), cudaMemcpyDeviceToHost ) );

		for( i = 0; i < N; i++ )
			TEST( fabs( res[i] - (v1[i] - ALFA * v2[i]) ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	if( !strcmp( function, "dscalv" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		dscalv( ALFA, dev_v1, dev_res, N, 0 );

		CUDA( cudaMemcpy( res, dev_res, N * sizeof( double ), cudaMemcpyDeviceToHost ) );

		for( i = 0; i < N; i++ )
			TEST( fabs( res[i] - ALFA * v1[i] ) < EPSILON );

		status = cup_finish( );
		TEST( status == CUP_SUCCESS );

		END_TEST();
	}

	TEST( 0 );
}

/********************************************
 * Private functions                        *
 ********************************************/
static void init_vec( double* v, int N )
{
	int i;

	assert( N > 0 );
	assert( v != NULL );

	for( i = 0; i < N; i++ )
		v[i] = (double) (random() % 10000 );
}
