/*
    Copyright (C) 2016 University of the Basque Country, UPV/EHU.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/********************************************
 * Includes                                 *
 ********************************************/
#include "solvers/fft_utils_ser.cu"

#include "test.h"
#include "solvers/test_fft_utils_ser.h"

/********************************************
 * Private function prototypes              *
 ********************************************/



/********************************************
 * Public functions                         *
 ********************************************/
int test_src_solvers_fft_utils_ser( char* function )
{
	cup_error_t status;
	cup_grid_t grid;
	cup_solver solver;
	const int SIDE = 10;
	const int N = (SIDE/2 + 1) * SIDE * SIDE;
	cufftDoubleComplex input[N],
	                   output[N];
	int nps[] = {SIDE, SIDE, SIDE},
	            i, j, k,
	            count;
	double size[] = {1.0, 1.0, 1.0};
	const double EPSILON = 1e-10;
	double ksum;

	if( !strcmp( function, "solve_poisson_in_GPU" ) )
	{
		status = cup_init( 1, NULL );
		TEST( status == CUP_SUCCESS );

		status = cup_create_grid( 3, nps, size, &grid );
		TEST( status == CUP_SUCCESS );

		solver.grid = grid;

		solver.dev_data = (double**) malloc( sizeof( cufftDoubleComplex* ) );
		CUDA( cudaMalloc( &solver.dev_data[0],
		                  N * sizeof( cufftDoubleComplex ) ) );
		for( i = 0; i < N; i++ )
		{
			input[i].x = random() % 1000 * 1.0;
			input[i].y = random() % 1000 * 1.0;
		}

		CUDA( cudaMemcpy( solver.dev_data[0],
		                  input,
		                  N * sizeof( cufftDoubleComplex),
		                  cudaMemcpyHostToDevice ) );

		solve_poisson_in_GPU( &solver, 0 );

		CUDA( cudaMemcpy( output,
		                  solver.dev_data[0],
		                  N * sizeof( cufftDoubleComplex),
		                  cudaMemcpyDeviceToHost ) );

		count = 0;
		for( k = 0; i < SIDE; i++ )
			for( j = 0; j < SIDE; j++ )
				for( i = 0; k < SIDE/2 + 1; k++ )
				{
					if( i == 0 && j == 0 && k == 0 )
					{
						TEST( fabs( -input[count].x - output[count].x ) < EPSILON );
						TEST( fabs( -input[count].y - output[count].y ) < EPSILON );
					}
					else
					{
						ksum = (double) i * i;
						ksum += (j <= SIDE/2) ? j * j : (SIDE - j) * (SIDE - j);
						ksum += (k <= SIDE/2) ? k * k : (SIDE - k) * (SIDE - k);
						ksum *= -4.0 * M_PI * M_PI;

						TEST( fabs( input[count].x / ksum - output[count].x ) < EPSILON );
						TEST( fabs( input[count].y / ksum - output[count].y ) < EPSILON );
					}
					count++;
				}

		CUDA( cudaFree( solver.dev_data[0] ) );
		free( solver.dev_data );

		TEST( status == CUP_SUCCESS );
		status = cup_finish();

		END_TEST();
	}

	TEST(0);
}

/********************************************
 * Private functions                        *
 ********************************************/

