/*
    Copyright (C) 2016 University of the Basque Country, UPV/EHU.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/** \file
 * Grid management functions.
 */

/********************************************
 * Includes                                 *
 ********************************************/
#include <stdlib.h>
#include <string.h>

#include "core/globals.h"

#include "grid.h"

/********************************************
 * Exported functions                        *
 ********************************************/
#ifdef __cplusplus
extern "C" {
#endif

/**	\ingroup public
 * Creates a grid element of type cup_grid_t.
 * \param[in] dim The number of dimensions of the grid.
 * \param[in] nps An array of \p dim elements with the number of points in each
 *  dimension of the grid.
 * \param[in] size An array of \p dim elements with the size each
 *  dimension of the grid.
 * \param[out] grid A cup_grid_t pointer. On success it will point to the created
 *  element.
 * \return CUP_SUCCESS on success, or CUP_INVALID_ARGUMENT if any input argument
 * is invalid.
 */
cup_error_t cup_create_grid( int           dim,
                             const int*    nps,
                             const double* size,
                             cup_grid_t*   grid)
{
	struct cup_grid* _grid;
	int i;
	long np;

	assert( grid != NULL );
	assert( nps != NULL );
	assert( size != NULL );

	if( dim <= 0 )
	{
		print_msg( ERROR, "The dimensionality must be a positive value." );
		return CUP_INVALID_ARGUMENT;
	}

	for( i = 0; i < dim; i++ )
		if( nps[i] <= 0 )
		{
			print_msg( ERROR, "The number of points of every dimension must be a "
			                  "positive value." );
			return CUP_INVALID_ARGUMENT;
		}

	MALLOC( _grid, 1, struct cup_grid );

	np = nps[0];
	for( i = 1; i < dim; i++ )
		np *= nps[i];

	_grid->dim = dim;
	_grid->np = np;
	MALLOC( _grid->nps, dim, int );
	memcpy( _grid->nps, nps, dim * sizeof( int ) );
	MALLOC( _grid->size, dim, double );
	memcpy( _grid->size, size, dim * sizeof( double ) );

	*grid = _grid;

	return CUP_SUCCESS;
}

/**	\ingroup public
 * Releases the memory used by a grid created with the \p cup_create_grid
 * function.
 * \param[in,out] grid The grid to be destroyed.
 */
void cup_destroy_grid( cup_grid_t grid )
{
	free( grid->nps );
	free( grid );
}

#ifdef __cplusplus
} // extern "C"
#endif

/********************************************
 * Public functions                         *
 ********************************************/
/**
 * Copy a grid structure to a new structure. The function allocates the
 * necessary memory space.
 * \param[out] out A pointer that will point to the copied grid.
 * \param[in] in A pointer to the input grid.
 * \return CUP_SUCCESS, or CUP_OUT_OF_HOST_MEM.
 */
cup_error_t copy_grid( struct cup_grid** out, const struct cup_grid* in )
{
	struct cup_grid* _out;

	assert( out != NULL && in != NULL );

	MALLOC( _out, 1, struct cup_grid );

	_out->dim = in->dim;
	MALLOC( _out->nps, _out->dim, int );
	memcpy( _out->nps, in->nps, _out->dim * sizeof( int ) );
	MALLOC( _out->size, _out->dim, double );
	memcpy( _out->size, in->size, _out->dim * sizeof( double ) );
	_out->np = in->np;

	*out = _out;

	return CUP_SUCCESS;
}
