/*
    Copyright (C) 2016 University of the Basque Country, UPV/EHU.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef GLOBALS_H_
#define GLOBALS_H_

/********************************************
 * Includes                                 *
 ********************************************/
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#ifdef USE_CUBLAS
#include "cublas_v2.h"
#endif
#if VERBOSE != 0
#include "nvToolsExt.h"
#endif
#include <cufft.h>

/********************************************
 * Macros                                   *
 ********************************************/
// Message printing
#define PRINT_NAME    "cuPoisson"
#define LOG_FILE_NAME "cuPoisson"

#ifndef DEBUG
#define DEBUG 0
#endif

#define ERROR           0
#define TRACE           1
#define WARNING         2
#define INFO            3
#define NUM_PRINT_MODES 4

#if __CUDA_ARCH__ >= 200
#define MAX_NUMBER_OF_THREADS 1024
#else
#define MAX_NUMBER_OF_THREADS 512
#endif

//Macro for handling the values returned by CUDA functions.
#if DEBUG != 0
#define CUDA(x) \
{ \
	x; \
	cudaThreadSynchronize(); \
	cudaError_t err = cudaGetLastError(); \
	if( cudaSuccess != err ) \
	{ \
		print_msg( ERROR, \
		           "CUDA Error on call \"%s\": %s\n\tLine: %d, File: %s\n", \
		           #x, cudaGetErrorString( err ), __LINE__, __FILE__); \
		return CUP_CUDA_ERROR; \
	} \
}
#else
#define CUDA(x) \
{ \
	cudaError_t err = x; \
	if( cudaSuccess != err ) \
	{ \
		print_msg( ERROR, \
		           "CUDA Error on call \"%s\": %s\n\tLine: %d, File: %s\n", \
		           #x, cudaGetErrorString( err ), __LINE__, __FILE__); \
		return CUP_CUDA_ERROR; \
	} \
}
#endif

// Macro that propagates CUP errors to calling functions.
#define THROW_CUP_ERROR( x ) \
{ \
	cup_error_t err = x; \
	if( err != CUP_SUCCESS ) \
		return err; \
}

//Macro that checks whether the library is initialized and returns otherwise.
#define CHECK_INIT() if( !cup_is_init() )  return CUP_NOT_INITIALIZED;

/**
 * Macro that allocates CPU memory. Makes calling to malloc easier since it
 * reduces explicit castings and multiplications.
 * \param[out] ptr The pointer that will point to the allocated memory.
 * \param[in] n The size of the allocated memory space in elements.
 * \param[in] type The data type of the elements.
 */
#define MALLOC( ptr, n, type ) \
{ \
	(ptr) = (type*) malloc( (n) * sizeof( type ) ); \
	if( (ptr) == NULL ) \
	{ \
		print_msg( ERROR, \
		           "cuPoisson: Host memory allocation error.\n\tLine: %d, " \
		           "File: %s, N: %d, Type: %s\n", \
		           __LINE__, __FILE__, n, #type ); \
		return( CUP_OUT_OF_HOST_MEM ); \
	} \
}

// Macro to securely call to POSIX thread related functions.
#define PTHREAD(x) \
{ \
	int st = (x); \
	if( st != 0 ) \
	{ \
		print_msg( ERROR, \
		           "Pthread error on call \"%s\"\n\tLine: %d, File: %s\n", \
		           #x, __LINE__, __FILE__); \
		return CUP_THREAD_ERROR; \
	} \
}

// Macro tu securely call to MPI related functions.
#ifdef MOD_MPI
#define MPI_CALL( x ) \
{ \
	int st = ( x ); \
	if( st != MPI_SUCCESS ) \
	{ \
		print_msg( ERROR, \
		           "MPI error on call \"%s\"\n\tLine: %d, File: %s\n", \
		           #x, __LINE__, __FILE__); \
		return CUP_MPI_ERROR; \
	} \
}
#endif

// Macros to control CUDA profiling ranges.
#if VERBOSE != 0
#define PROFILE_PUSH( name ) nvtxRangePushA( name )
#define PROFILE_POP() nvtxRangePop()
#else
#define PROFILE_PUSH( name )
#define PROFILE_POP()
#endif

/********************************************
 * Data definitions (anticipated)          *
 ********************************************/
struct config
{
	int          verbose_level;
	FILE*        msg_output[NUM_PRINT_MODES];
	int          mpi_nonblocking;
	unsigned int mpi_segment_size;
};

struct cup_global_info
{
	struct config          config;
	int                    num_devices;
	int*                   devices;
	struct cudaDeviceProp* device_info;
#ifdef USE_CUBLAS
	int                    use_cublas;
	cublasHandle_t*        cublas_handles;
#endif
	pid_t                  pid;
};

/********************************************
 * Macros (continued)                       *
 ********************************************/
/** Macro to print log messages.
 * \param[in] type The message type.
 * \param[in] msg The message to be print in printf format.
 * \param[in] ... Parameters used in \p msg.
 */
#define print_msg( type, msg, ... ) \
{ \
	if( type == ERROR ) \
		fprintf( global_info.config.msg_output[type], \
		         "%s: (%06d) [%s] " msg "\n%s: (%06d)\tLine: %d, File: %s\n", \
		         PRINT_NAME, global_info.pid, #type, ##__VA_ARGS__, \
		         PRINT_NAME, global_info.pid, __LINE__, __FILE__ );\
	else if( (DEBUG && type == TRACE) || \
			 (VERBOSE && type - 1 <= global_info.config.verbose_level) ) \
		fprintf( global_info.config.msg_output[type], \
		         "%s: (%06d) [%s] " msg "\n", \
		         PRINT_NAME, global_info.pid, #type, ##__VA_ARGS__ ); \
	if( DEBUG ) \
		fflush( global_info.config.msg_output[type] ); \
} \

/********************************************
 * Global variables                         *
 ********************************************/
extern int                    glob_is_init;
extern struct cup_global_info global_info;

#endif /* GLOBALS_H_ */
